package gov.va.genisis2.dao.impl;

import java.util.List;

import javax.persistence.NoResultException;
import javax.persistence.Query;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Join;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import gov.va.genisis2.common.enums.CommonEnum;
import gov.va.genisis2.common.enums.UserStatusEnum;
import gov.va.genisis2.dao.IUserManagementDao;
import gov.va.genisis2.exceptions.ErrorEnum;
import gov.va.genisis2.exceptions.GenisisDAOException;
import gov.va.genisis2.model.User;
import gov.va.genisis2.model.UserCount;
import gov.va.genisis2.model.UserRoleType;

/**
 * The Class UserManagementDAO.
 *
 * The UserManagement data access object (DAO) is an object that provides an
 * abstract interface to some type of database or other persistence mechanism.
 * By mapping application calls to the persistence layer, UserManagement DAO
 * provide some specific data operations without exposing details of the
 * database.
 */
@Repository
@Transactional(value = "transactionManager")
public class UserManagementDao extends AbstactHibernateDao implements IUserManagementDao {

	/** The LOGGER. */
	private static final Logger LOGGER = LoggerFactory.getLogger(UserManagementDao.class);
	private static final String USERSTATUS_UPDATE_SQL = "UPDATE User SET User_Active = :disableStatus, ModifiedOn = GETDATE() WHERE user_name NOT IN (:usersList) AND User_Active = :activeStatus";

	@Autowired
	private SessionFactory sessionFactory;

	/**
	 * This method is used to getUserDetailsByEmail.
	 * 
	 * @param email
	 *            The email.
	 * @return User This returns User.
	 */
	@Override
	public User getUserDetailsByEmail(String email) throws GenisisDAOException {
		User user = null;

		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<User> query = criteriaBuilder.createQuery(User.class);
			Root<User> root = query.from(User.class);
			Predicate condition = criteriaBuilder.equal(root.get(CommonEnum.EMAIL_ID.getText()), email);
			query.where(condition);
			user = session.createQuery(query).getSingleResult();
		} catch (NoResultException e) {
			// Swallow exception
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETUSERDETAILSBYEMAIL.getErrorMessage(), e);
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETUSERDETAILSBYEMAIL.getErrorMessage(), ex);
			throw new GenisisDAOException(ex.getClass().getSimpleName(), ex);
		}

		return user;
	}

	/**
	 * This method is used to getUserDetailsById.
	 * 
	 * @param id
	 *            The id.
	 * @return User This returns user.
	 */
	@Override
	public User getUserDetailsById(int id) throws GenisisDAOException {
		User user = null;

		try {
			Session session = sessionFactory.getCurrentSession();
			user = session.get(User.class, id);
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETUSERDETAILSBYID.getErrorMessage(), ex);
			throw new GenisisDAOException(ex.getClass().getSimpleName(), ex);
		}

		return user;
	}

	/**
	 * This method is used to getUserRole.
	 * 
	 * @param uid
	 *            The uid.
	 * @return String This returns userRole.
	 */
	@Override
	public String getUserRole(int uid) throws GenisisDAOException {
		String userRole = StringUtils.EMPTY;
		UserRoleType userRoleType = null;

		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<UserRoleType> query = criteriaBuilder.createQuery(UserRoleType.class);
			Root<UserRoleType> root = query.from(UserRoleType.class);
			Predicate condition = criteriaBuilder.equal(root.get(CommonEnum.USER.getText()).get(CommonEnum.USER_ID.getText()), uid);
			query.where(condition);
			userRoleType = session.createQuery(query).getSingleResult();
		} catch (NoResultException e) {
			// Swallow exception
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETUSERROLE.getErrorMessage(), e);
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETUSERROLE.getErrorMessage(), ex);
			throw new GenisisDAOException(ex.getClass().getSimpleName(), ex);
		}

		if (userRoleType != null) {
			userRole = userRoleType.getRoleType().getRoleName();
		}

		return userRole;
	}

	/**
	 * This method is used to get user details by username.
	 * 
	 * @param username
	 *            The username.
	 * @return User This returns User.
	 */
	@Override
	public User getUserDetailsByUsername(String username) throws GenisisDAOException {
		User user = null;

		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<User> query = criteriaBuilder.createQuery(User.class);
			Root<User> root = query.from(User.class);
			Predicate condition = criteriaBuilder.equal(root.get(CommonEnum.USER_NAME.getText()), username);
			query.where(condition);
			user = session.createQuery(query).getSingleResult();
		} catch (NoResultException e) {
			// Swallow exception
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETUSERDETAILSBYUSERNAME.getErrorMessage(), e);
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETUSERDETAILSBYUSERNAME.getErrorMessage(), ex);
			throw new GenisisDAOException(ex.getClass().getSimpleName(), ex);
		}

		return user;
	}

	@Override
	public List<UserCount> getUserCountOnRole() throws GenisisDAOException {
		List<UserCount> userCounts = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<UserCount> query = criteriaBuilder.createQuery(UserCount.class);
			Root<UserRoleType> root = query.from(UserRoleType.class);
			query.select(criteriaBuilder.construct(UserCount.class, root.get(CommonEnum.ROLE_TYPE_ID.getText()), criteriaBuilder.count(root)));
			query.where(criteriaBuilder.equal(root.get(CommonEnum.USER.getText()).get(CommonEnum.USER_ACTIVE.getText()), UserStatusEnum.ACTIVE.getId()));
			query.groupBy(root.get(CommonEnum.ROLE_TYPE_ID.getText()));
			userCounts = session.createQuery(query).getResultList();
		} catch (NoResultException e) {
			// Swallow exception
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETUSERCOUNTONROLE.getErrorMessage(), e);
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETUSERCOUNTONROLE.getErrorMessage(), ex);
			throw new GenisisDAOException(ex.getClass().getSimpleName(), ex);
		}
		return userCounts;
	}
	
	@Override
	public List<User> getUserDetailsByRoleTypeId(int roleTypeId) throws GenisisDAOException {
		List<User> users = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<User> criteriaQuery = criteriaBuilder.createQuery(User.class);
			Root<User> userRoot = criteriaQuery.from(User.class);
			Join<User, UserRoleType> userRoleType = userRoot.join(CommonEnum.USER_ROLE_TYPES.getText());
			Predicate userCondition = criteriaBuilder.equal(userRoleType.get(CommonEnum.USER_ID.getText()), userRoot.get(CommonEnum.USER_ID.getText()));
			Predicate userRoleTypeCondition = criteriaBuilder.equal(userRoleType.get(CommonEnum.ROLE_TYPE_ID.getText()), roleTypeId);
			Predicate activeUsers = criteriaBuilder.equal(userRoot.get(CommonEnum.USER_ACTIVE.getText()), UserStatusEnum.ACTIVE.getId());
			Predicate conditions = criteriaBuilder.and(userCondition, userRoleTypeCondition, activeUsers);
			criteriaQuery.where(conditions);
			users = session.createQuery(criteriaQuery).getResultList();
		} catch (NoResultException e) {
			// Swallow exception
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETUSERDETAILSBYROLEID.getErrorMessage(), e);
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETUSERDETAILSBYROLEID.getErrorMessage(), ex);
			throw new GenisisDAOException(ex.getClass().getSimpleName(), ex);
		}
		return users;
	}
	
	@Override
	public List<User> getAllUsers() throws GenisisDAOException {
		List<User> users = null;

		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<User> query = criteriaBuilder.createQuery(User.class);
			Root<User> root = query.from(User.class);
			query.select(root);
			users = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.USER_DAO_EXP_GETALLUSERS.getErrorMessage(), ex);
			throw new GenisisDAOException(ex.getClass().getSimpleName(), ex);
		}

		return users;
	}
	
	@Override
	public User saveOrUpdate(User user) throws GenisisDAOException {
		if (user == null) {
			throw new GenisisDAOException("user entity is null");
		}
		
		User persistentUser;
		try {
			Session session = sessionFactory.getCurrentSession();
			if (user.getUserId() <= 0) {
				user.setUserId(getMaxRowValue(session, User.class, CommonEnum.USER_ID.getText()));
			}
			persistentUser = (User) updateData(session, user);
		} catch (Exception ex) {
			LOGGER.error("Exception occured on createuser.", ex);
			throw new GenisisDAOException("Exception occured on createuser.", ex);
		}
		
		return persistentUser;
	}
	
	@Override
	public int updateUserStatus(List<String> usersList) throws GenisisDAOException {
		int result = 0;
		try {
			Session session = sessionFactory.getCurrentSession();
			Query query = session.createQuery(USERSTATUS_UPDATE_SQL);
			query.setParameter("disableStatus", UserStatusEnum.DISABLE.getId());
			query.setParameter("usersList", usersList);
			query.setParameter("activeStatus", UserStatusEnum.ACTIVE.getId());
			result = query.executeUpdate();
		} catch (Exception ex) {
			LOGGER.error("Exception occured on updateUserStatus", ex);
			throw new GenisisDAOException("Exception occured on updateUserStatus", ex);
		}
		
		return result;
	}

	/**
	 * @param sessionFactory the sessionFactory to set
	 */
	public void setSessionFactory(SessionFactory sessionFactory) {
		this.sessionFactory = sessionFactory;
	}
}
